/*
 * Decompiled with CFR 0.152.
 */
package ee.lnsolutions.tengine.service;

import ee.lnsolutions.tengine.service.XRechnungTransformService;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.URIResolver;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import net.sf.saxon.TransformerFactoryImpl;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.tidy.Tidy;
import org.xhtmlrenderer.pdf.ITextRenderer;
import org.xml.sax.InputSource;

public class XRechnungTransformService {
    private static final Logger logger = LoggerFactory.getLogger(XRechnungTransformService.class);

    public boolean isXRechnungXml(InputStream input) throws Exception {
        logger.debug("Checking if XML is XRechnung format");
        String xmlContent = this.cleanBomFromInputStream(input);
        String invoiceType = this.identifyInvoiceType(xmlContent);
        boolean isSupported = "UBL Invoice".equals(invoiceType) || "UBL CreditNote".equals(invoiceType) || "CII Invoice".equals(invoiceType) || "ISO20022 pain.001".equals(invoiceType);
        logger.debug("Invoice type detected: {}, is XRechnung: {}", (Object)invoiceType, (Object)isSupported);
        return isSupported;
    }

    public String transformToHtml(InputStream input) throws Exception {
        logger.info("Starting XRechnung to HTML transformation");
        String xmlContent = this.cleanBomFromInputStream(input);
        StringWriter intermediateXml = this.transformToIntermediate(xmlContent);
        StringWriter htmlOutput = this.applyXSLT(intermediateXml, "xsl/xrechnung-html.xsl");
        String cleanHtml = this.cleanBomFromString(htmlOutput.toString());
        logger.info("XRechnung transformation completed successfully");
        return cleanHtml;
    }

    public byte[] transformToPdf(InputStream input) throws Exception {
        logger.info("Starting XRechnung to PDF transformation");
        String htmlContent = this.transformToHtml(input);
        return this.convertHtmlToPdf(htmlContent);
    }

    public byte[] convertHtmlToPdf(String htmlContent) throws Exception {
        byte[] byArray;
        logger.debug("Converting HTML to PDF using Flying Saucer, HTML length: {} bytes", (Object)htmlContent.length());
        Tidy tidy = new Tidy();
        tidy.setXHTML(true);
        tidy.setQuiet(true);
        tidy.setShowWarnings(false);
        tidy.setInputEncoding("UTF-8");
        tidy.setOutputEncoding("UTF-8");
        Document xhtmlDoc = tidy.parseDOM((InputStream)new ByteArrayInputStream(htmlContent.getBytes(StandardCharsets.UTF_8)), null);
        ITextRenderer renderer = new ITextRenderer();
        renderer.setDocument(xhtmlDoc, null);
        renderer.layout();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            renderer.createPDF((OutputStream)out);
            byte[] pdfBytes = out.toByteArray();
            logger.debug("PDF conversion completed, PDF size: {} bytes", (Object)pdfBytes.length);
            byArray = pdfBytes;
        }
        catch (Throwable throwable) {
            try {
                try {
                    out.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                logger.error("Flying Saucer conversion failed", (Throwable)e);
                throw new Exception("HTML to PDF conversion failed: " + e.getMessage(), e);
            }
        }
        out.close();
        return byArray;
    }

    private StringWriter transformToIntermediate(String xmlContent) throws TransformerException, IOException {
        try {
            Transformer transformer;
            String workingXml = xmlContent;
            String invoiceType = this.identifyInvoiceType(xmlContent);
            if ("ISO20022 pain.001".equals(invoiceType)) {
                logger.debug("Detected ISO20022 pain.001.001.03 document; converting to UBL Invoice");
                workingXml = this.transformXml(workingXml, "xsl/pain001-to-ubl-invoice.xsl");
                invoiceType = this.identifyInvoiceType(workingXml);
                logger.debug("Invoice type after pain.001 conversion: {}", (Object)invoiceType);
            }
            String intermediateXslFile = "xsl/ubl-invoice-xr.xsl";
            if (invoiceType.equals("UBL Invoice")) {
                intermediateXslFile = "xsl/ubl-invoice-xr.xsl";
            } else if (invoiceType.equals("UBL CreditNote")) {
                intermediateXslFile = "xsl/ubl-creditnote-xr.xsl";
            } else if (invoiceType.equals("CII Invoice")) {
                intermediateXslFile = "xsl/cii-xr.xsl";
            } else {
                logger.warn("Unsupported invoice type detected: {}", (Object)invoiceType);
                throw new TransformerException("Unsupported invoice type: " + invoiceType);
            }
            logger.debug("Invoice Type: {}, Intermediate XSL File: {}", (Object)invoiceType, (Object)intermediateXslFile);
            URL xslFileUrl = this.getClass().getClassLoader().getResource(intermediateXslFile);
            if (xslFileUrl == null) {
                throw new IOException("Could not locate intermediate XSL file: " + intermediateXslFile);
            }
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            factory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            String cleanXml = this.sanitizeXml(workingXml);
            Document doc = builder.parse(new InputSource(new StringReader(cleanXml)));
            DOMSource source = new DOMSource(doc);
            JarResourceURIResolver uriResolver = new JarResourceURIResolver(xslFileUrl.toString());
            TransformerFactoryImpl transformerFactory = new TransformerFactoryImpl();
            transformerFactory.setFeature("http://javax.xml.XMLConstants/feature/secure-processing", true);
            transformerFactory.setURIResolver((URIResolver)uriResolver);
            try (InputStream xslStream = xslFileUrl.openStream();){
                transformer = transformerFactory.newTransformer(new StreamSource(xslStream));
            }
            transformer.setOutputProperty("encoding", "UTF-8");
            StringWriter sw = new StringWriter();
            transformer.transform(source, new StreamResult(sw));
            this.validateXml(sw.toString());
            return sw;
        }
        catch (Exception e) {
            logger.error("Error in intermediate transformation", (Throwable)e);
            if (e instanceof TransformerException) {
                throw (TransformerException)e;
            }
            throw new TransformerException("Could not transform to intermediate XML: " + e.getMessage(), e);
        }
    }

    private StringWriter applyXSLT(StringWriter input, String xslPath) throws TransformerException, IOException {
        try {
            Transformer transformer;
            URL xslFileUrl = this.getClass().getClassLoader().getResource(xslPath);
            if (xslFileUrl == null) {
                throw new IOException("Could not locate XSL file: " + xslPath);
            }
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            factory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(new InputSource(new StringReader(this.sanitizeXml(input.toString()))));
            DOMSource source = new DOMSource(doc);
            TransformerFactoryImpl transformerFactory = new TransformerFactoryImpl();
            transformerFactory.setFeature("http://javax.xml.XMLConstants/feature/secure-processing", true);
            transformerFactory.setURIResolver((URIResolver)new JarResourceURIResolver(xslFileUrl.toString()));
            try (InputStream xslStream = xslFileUrl.openStream();){
                transformer = transformerFactory.newTransformer(new StreamSource(xslStream));
            }
            transformer.setOutputProperty("encoding", "UTF-8");
            transformer.setOutputProperty("method", "html");
            transformer.setOutputProperty("omit-xml-declaration", "yes");
            StringWriter output = new StringWriter();
            transformer.transform(source, new StreamResult(output));
            return output;
        }
        catch (Exception e) {
            logger.error("Error applying XSLT transformation", (Throwable)e);
            if (e instanceof TransformerException) {
                throw (TransformerException)e;
            }
            throw new TransformerException("Could not apply XSLT transformation: " + e.getMessage(), e);
        }
    }

    private void validateXml(String xmlContent) throws Exception {
        if (xmlContent == null || xmlContent.trim().isEmpty()) {
            throw new IllegalArgumentException("XML content is empty or null");
        }
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setValidating(false);
        factory.setNamespaceAware(true);
        factory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
        String sanitized = this.sanitizeXml(xmlContent);
        try {
            DocumentBuilder builder = factory.newDocumentBuilder();
            builder.parse(new InputSource(new StringReader(sanitized)));
        }
        catch (Exception e) {
            logger.error("XML Validation Error: {}", (Object)e.getMessage());
            logger.error("Problematic XML Content (sanitized start): {}", (Object)sanitized.substring(0, Math.min(200, sanitized.length())));
            throw e;
        }
    }

    private String sanitizeXml(String xmlContent) {
        int declEnd;
        int firstBracket;
        if (xmlContent == null) {
            return null;
        }
        Object sanitized = xmlContent;
        if (((String)sanitized).startsWith("\ufeff")) {
            sanitized = ((String)sanitized).substring(1);
        }
        if ((firstBracket = ((String)(sanitized = ((String)sanitized).trim())).indexOf(60)) > 0) {
            sanitized = ((String)sanitized).substring(firstBracket);
        }
        if (((String)sanitized).startsWith("<?xml") && (declEnd = ((String)sanitized).indexOf("?>")) != -1) {
            String beforeDecl = ((String)sanitized).substring(0, declEnd + 2);
            String afterDecl = ((String)sanitized).substring(declEnd + 2);
            String cleanedAfterDecl = afterDecl.replaceFirst("^[^<]+", "");
            sanitized = beforeDecl + cleanedAfterDecl;
        }
        return sanitized;
    }

    private String identifyInvoiceType(String xmlContent) throws Exception {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        factory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
        DocumentBuilder builder = factory.newDocumentBuilder();
        String sanitized = this.sanitizeXml(xmlContent);
        Document document = builder.parse(new InputSource(new StringReader(sanitized)));
        String rootElement = document.getDocumentElement().getLocalName();
        String namespaceURI = document.getDocumentElement().getNamespaceURI();
        if ("Invoice".equals(rootElement) && "urn:oasis:names:specification:ubl:schema:xsd:Invoice-2".equals(namespaceURI)) {
            return "UBL Invoice";
        }
        if ("CreditNote".equals(rootElement) && "urn:oasis:names:specification:ubl:schema:xsd:CreditNote-2".equals(namespaceURI)) {
            return "UBL CreditNote";
        }
        if ("CrossIndustryInvoice".equals(rootElement) && "urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100".equals(namespaceURI)) {
            return "CII Invoice";
        }
        if ("Document".equals(rootElement) && "urn:iso:std:iso:20022:tech:xsd:pain.001.001.03".equals(namespaceURI)) {
            return "ISO20022 pain.001";
        }
        return "Unknown XML Type";
    }

    private String transformXml(String xmlContent, String xslClasspathLocation) throws Exception {
        URL xslFileUrl = this.getClass().getClassLoader().getResource(xslClasspathLocation);
        if (xslFileUrl == null) {
            throw new IOException("Could not locate XSL file: " + xslClasspathLocation);
        }
        JarResourceURIResolver uriResolver = new JarResourceURIResolver(xslFileUrl.toString());
        TransformerFactoryImpl transformerFactory = new TransformerFactoryImpl();
        transformerFactory.setFeature("http://javax.xml.XMLConstants/feature/secure-processing", true);
        transformerFactory.setURIResolver((URIResolver)uriResolver);
        try (InputStream xslStream = xslFileUrl.openStream();){
            Transformer transformer = transformerFactory.newTransformer(new StreamSource(xslStream));
            transformer.setOutputProperty("encoding", "UTF-8");
            String sanitizedInput = this.sanitizeXml(xmlContent);
            StringWriter output = new StringWriter();
            transformer.transform(new StreamSource(new StringReader(sanitizedInput)), new StreamResult(output));
            String string = output.toString();
            return string;
        }
    }

    private String cleanBomFromInputStream(InputStream input) throws IOException {
        byte[] bytes = IOUtils.toByteArray((InputStream)input);
        if (bytes.length >= 3 && bytes[0] == -17 && bytes[1] == -69 && bytes[2] == -65) {
            logger.debug("Removing UTF-8 BOM from XRechnung XML content");
            byte[] cleaned = new byte[bytes.length - 3];
            System.arraycopy(bytes, 3, cleaned, 0, cleaned.length);
            return new String(cleaned, StandardCharsets.UTF_8);
        }
        if (bytes.length >= 2 && bytes[0] == -2 && bytes[1] == -1) {
            logger.debug("Removing UTF-16 BE BOM from XRechnung XML content");
            byte[] cleaned = new byte[bytes.length - 2];
            System.arraycopy(bytes, 2, cleaned, 0, cleaned.length);
            return new String(cleaned, StandardCharsets.UTF_16BE);
        }
        if (bytes.length >= 2 && bytes[0] == -1 && bytes[1] == -2) {
            logger.debug("Removing UTF-16 LE BOM from XRechnung XML content");
            byte[] cleaned = new byte[bytes.length - 2];
            System.arraycopy(bytes, 2, cleaned, 0, cleaned.length);
            return new String(cleaned, StandardCharsets.UTF_16LE);
        }
        return new String(bytes, StandardCharsets.UTF_8);
    }

    private String cleanBomFromString(String input) {
        if (input == null) {
            return null;
        }
        if (input.startsWith("\ufeff")) {
            logger.debug("Removing BOM character from HTML output");
            return input.substring(1);
        }
        return input;
    }
}

